"""
Blender Operators for Mocai Add-on
Handles video upload, status polling, and GLB import
"""

import bpy
import os
import tempfile
from bpy.types import Operator
from .api_client import MocaiAPIClient


class MOCAI_OT_authenticate(Operator):
    """Authenticate with Mocai API"""
    bl_idname = "mocai.authenticate"
    bl_label = "Authenticate"
    bl_description = "Authenticate with your Mocai API key"

    def execute(self, context):
        props = context.scene.mocai

        if not props.api_key:
            self.report({'ERROR'}, "Please enter your API key")
            return {'CANCELLED'}

        # Create API client and authenticate
        client = MocaiAPIClient(props.api_key)

        if client.authenticate():
            self.report({'INFO'}, "Authentication successful!")
            props.status_message = "Authenticated successfully"
            props.is_authenticated = True
            return {'FINISHED'}
        else:
            # Check if it was a timeout or connection error
            try:
                # Try a simple request to check connectivity
                import requests
                requests.get("https://api.mocai.ai", timeout=5)
                self.report({'ERROR'}, "Authentication failed. Check your API key.")
                props.status_message = "Authentication failed - Invalid API key"
            except requests.exceptions.Timeout:
                self.report({'ERROR'}, "Connection timed out. Please check your internet connection.")
                props.status_message = "Connection timed out"
            except requests.exceptions.ConnectionError:
                self.report({'ERROR'}, "Could not connect to Mocai servers. Please check your internet connection.")
                props.status_message = "Connection error"
            except:
                self.report({'ERROR'}, "Authentication failed. Check your API key.")
                props.status_message = "Authentication failed"

            props.is_authenticated = False
            return {'CANCELLED'}


class MOCAI_OT_upload_video(Operator):
    """Upload video to Mocai for processing"""
    bl_idname = "mocai.upload_video"
    bl_label = "Upload Video"
    bl_description = "Upload video to Mocai for processing"

    def execute(self, context):
        props = context.scene.mocai

        # Validate inputs
        if not props.api_key:
            self.report({'ERROR'}, "Please enter your API key")
            return {'CANCELLED'}

        if not props.video_path:
            self.report({'ERROR'}, "Please select a video file")
            return {'CANCELLED'}

        if not os.path.exists(props.video_path):
            self.report({'ERROR'}, f"Video file not found: {props.video_path}")
            return {'CANCELLED'}

        # Check file size (limit to 500MB)
        file_size = os.path.getsize(props.video_path)
        max_size = 500 * 1024 * 1024  # 500MB in bytes
        if file_size > max_size:
            size_mb = file_size / (1024 * 1024)
            self.report({'ERROR'}, f"File too large: {size_mb:.1f}MB. Maximum allowed size is 500MB")
            return {'CANCELLED'}

        # Create API client and authenticate
        client = MocaiAPIClient(props.api_key)

        if not client.authenticate():
            # Check if it was a timeout or connection error
            try:
                # Try a simple request to check connectivity
                import requests
                requests.get("https://api.mocai.ai", timeout=5)
                self.report({'ERROR'}, "Authentication failed. Check your API key.")
            except requests.exceptions.Timeout:
                self.report({'ERROR'}, "Connection timed out. Please check your internet connection.")
            except requests.exceptions.ConnectionError:
                self.report({'ERROR'}, "Could not connect to Mocai servers. Please check your internet connection.")
            except:
                self.report({'ERROR'}, "Authentication failed. Check your API key.")

            props.is_authenticated = False
            return {'CANCELLED'}

        # Update authenticated status
        props.is_authenticated = True

        # Upload video
        props.status_message = "Uploading video..."
        self.report({'INFO'}, "Uploading video...")

        result = client.upload_video(
            video_path=props.video_path,
            title=props.video_title,
            character=props.character,
            export_glb=props.export_glb,
            export_fbx=props.export_fbx,
            export_bvh=props.export_bvh,
            export_usd=props.export_usd,
            hand_motion_capture=props.hand_motion_capture,
            depth=props.depth
        )

        if result and result.get('videoId'):
            video_id = result['videoId']
            props.video_id = video_id
            props.is_processing = True
            props.processing_status = result.get('status', 'queued')
            props.status_message = f"Video uploaded! ID: {video_id}"

            self.report({'INFO'}, f"Video uploaded successfully! ID: {video_id}")

            # Start polling timer
            bpy.ops.mocai.poll_status()

            return {'FINISHED'}
        else:
            self.report({'ERROR'}, "Video upload failed")
            props.status_message = "Upload failed"
            return {'CANCELLED'}


class MOCAI_OT_poll_status(Operator):
    """Poll video processing status"""
    bl_idname = "mocai.poll_status"
    bl_label = "Check Status"
    bl_description = "Check video processing status"

    _timer = None
    _client = None
    _downloaded_files = set()  # Track which files have been downloaded

    def modal(self, context, event):
        props = context.scene.mocai

        if event.type == 'TIMER':
            # Check if we should stop polling
            if not props.is_processing or not props.video_id:
                self.cancel(context)
                return {'CANCELLED'}

            # Poll status
            if self._client is None:
                self._client = MocaiAPIClient(props.api_key)
                if not self._client.authenticate():
                    props.status_message = "Authentication failed"
                    self.cancel(context)
                    return {'CANCELLED'}

            status_data = self._client.get_video_status(props.video_id)

            if status_data:
                status = status_data.get('status', 'unknown')
                props.processing_status = status
                assets = status_data.get('assets', {})

                # Check for individual file availability and download as soon as available
                requested_files = []
                available_files = []

                if props.export_glb:
                    requested_files.append('glb')
                    if assets.get('glb') and assets['glb'].get('url'):
                        available_files.append('glb')

                if props.export_fbx:
                    requested_files.append('fbx')
                    if assets.get('fbx') and assets['fbx'].get('url'):
                        available_files.append('fbx')

                if props.export_bvh:
                    requested_files.append('bvh')
                    if assets.get('bvh') and assets['bvh'].get('url'):
                        available_files.append('bvh')

                if props.export_usd:
                    requested_files.append('usd')
                    if assets.get('usd') and assets['usd'].get('url'):
                        available_files.append('usd')

                # Download any newly available files
                newly_available = [f for f in available_files if f not in self._downloaded_files]
                if newly_available:
                    self._download_and_import_files(context, newly_available, assets)

                # Update status message
                if status == 'rendering':
                    props.status_message = f"Rendering... ({len(self._downloaded_files)}/{len(requested_files)} files ready)"
                elif status == 'processing':
                    props.status_message = f"Processing... ({len(self._downloaded_files)}/{len(requested_files)} files ready)"
                else:
                    props.status_message = f"Status: {status}"

                # Check if all requested files have been downloaded
                if requested_files and all(f in self._downloaded_files for f in requested_files):
                    props.is_processing = False
                    self.report({'INFO'}, f"All requested files downloaded and imported!")
                    props.status_message = f"Complete! Imported {len(self._downloaded_files)} file(s)"
                    self.cancel(context)
                    return {'FINISHED'}

                # Handle terminal statuses
                if status == 'processed':
                    # All processing complete, exit even if some files weren't available
                    props.is_processing = False
                    if self._downloaded_files:
                        self.report({'INFO'}, "Processing complete!")
                        props.status_message = f"Complete! Imported {len(self._downloaded_files)} file(s)"
                    else:
                        self.report({'WARNING'}, "Processing complete but no requested files were available")
                        props.status_message = "Complete - no files available"
                    self.cancel(context)
                    return {'FINISHED'}

                elif status == 'failed':
                    props.is_processing = False
                    message = status_data.get('message', 'Unknown error')
                    if self._downloaded_files:
                        self.report({'WARNING'}, f"Processing failed but {len(self._downloaded_files)} file(s) were imported")
                        props.status_message = f"Partial import: {message}"
                    else:
                        self.report({'ERROR'}, f"Processing failed: {message}")
                        props.status_message = f"Failed: {message}"
                    self.cancel(context)
                    return {'CANCELLED'}

                elif status == 'partially_completed':
                    # Some files available, rendering failed
                    props.is_processing = False
                    if self._downloaded_files:
                        self.report({'INFO'}, f"Imported {len(self._downloaded_files)} available file(s)")
                        props.status_message = f"Partially complete - {len(self._downloaded_files)} file(s) imported"
                    else:
                        self.report({'WARNING'}, "Processing partially completed but no files available")
                        props.status_message = "Partially complete - no files available"
                    self.cancel(context)
                    return {'FINISHED'}

        return {'PASS_THROUGH'}

    def _download_and_import_files(self, context, file_types, assets):
        """Download and import specified file types"""
        props = context.scene.mocai
        import tempfile

        temp_dir = tempfile.gettempdir()
        video_id = props.video_id

        for file_type in file_types:
            asset = assets.get(file_type)
            if not asset or not asset.get('url'):
                continue

            url = asset['url']
            # Use .usdz extension for USD files
            ext = 'usdz' if file_type == 'usd' else file_type
            filename = f"mocai_{video_id}.{ext}"
            file_path = os.path.join(temp_dir, filename)

            props.status_message = f"Downloading {file_type.upper()}..."
            self.report({'INFO'}, f"Downloading {file_type.upper()} file...")

            try:
                if self._client.download_asset(url, file_path):
                    # Import the file into Blender
                    if file_type == 'glb':
                        bpy.ops.import_scene.gltf(filepath=file_path)
                        self.report({'INFO'}, f"GLB imported successfully!")
                    elif file_type == 'fbx':
                        bpy.ops.import_scene.fbx(filepath=file_path)
                        self.report({'INFO'}, f"FBX imported successfully!")
                    elif file_type == 'bvh':
                        bpy.ops.import_anim.bvh(filepath=file_path)
                        self.report({'INFO'}, f"BVH imported successfully!")
                    elif file_type == 'usd':
                        bpy.ops.wm.usd_import(filepath=file_path)
                        self.report({'INFO'}, f"USDZ imported successfully!")

                    self._downloaded_files.add(file_type)
                else:
                    self.report({'WARNING'}, f"Failed to download {file_type.upper()}")
            except Exception as e:
                self.report({'WARNING'}, f"Failed to import {file_type.upper()}: {e}")

    def execute(self, context):
        props = context.scene.mocai

        if not props.video_id:
            self.report({'ERROR'}, "No video ID set")
            return {'CANCELLED'}

        # Reset downloaded files tracking for this new video
        self._downloaded_files = set()

        # Set up timer for polling
        wm = context.window_manager
        self._timer = wm.event_timer_add(props.poll_interval, window=context.window)
        wm.modal_handler_add(self)

        return {'RUNNING_MODAL'}

    def cancel(self, context):
        if self._timer:
            wm = context.window_manager
            wm.event_timer_remove(self._timer)
            self._timer = None
        self._client = None
        # Note: We intentionally don't reset _downloaded_files here so the user
        # can see what files were downloaded in the final status message


class MOCAI_OT_download_assets(Operator):
    """Download and import requested file formats"""
    bl_idname = "mocai.download_assets"
    bl_label = "Download Assets"
    bl_description = "Download and import the processed animation files"

    def execute(self, context):
        props = context.scene.mocai

        if not props.api_key or not props.video_id:
            self.report({'ERROR'}, "No video ID or API key")
            return {'CANCELLED'}

        # Get video status to get download URLs
        client = MocaiAPIClient(props.api_key)
        if not client.authenticate():
            self.report({'ERROR'}, "Authentication failed")
            return {'CANCELLED'}

        status_data = client.get_video_status(props.video_id)

        if not status_data:
            self.report({'ERROR'}, "Failed to get video status")
            return {'CANCELLED'}

        assets = status_data.get('assets', {})
        temp_dir = tempfile.gettempdir()
        video_id = props.video_id
        downloaded_files = []

        # Download GLB if requested and available
        if props.export_glb:
            glb_asset = assets.get('glb')
            if glb_asset and glb_asset.get('url'):
                glb_url = glb_asset['url']
                glb_filename = f"mocai_{video_id}.glb"
                glb_path = os.path.join(temp_dir, glb_filename)

                props.status_message = "Downloading GLB file..."
                self.report({'INFO'}, "Downloading GLB file...")

                if client.download_asset(glb_url, glb_path):
                    downloaded_files.append(('glb', glb_path))
                    props.download_path = glb_path

                    # Import GLB into Blender
                    try:
                        bpy.ops.import_scene.gltf(filepath=glb_path)
                        self.report({'INFO'}, f"GLB imported successfully!")
                    except Exception as e:
                        self.report({'WARNING'}, f"Failed to import GLB: {e}")

        # Download FBX if requested and available
        if props.export_fbx:
            fbx_asset = assets.get('fbx')
            if fbx_asset and fbx_asset.get('url'):
                fbx_url = fbx_asset['url']
                fbx_filename = f"mocai_{video_id}.fbx"
                fbx_path = os.path.join(temp_dir, fbx_filename)

                props.status_message = "Downloading FBX file..."
                self.report({'INFO'}, "Downloading FBX file...")

                if client.download_asset(fbx_url, fbx_path):
                    downloaded_files.append(('fbx', fbx_path))

                    # Import FBX into Blender
                    try:
                        bpy.ops.import_scene.fbx(filepath=fbx_path)
                        self.report({'INFO'}, f"FBX imported successfully!")
                    except Exception as e:
                        self.report({'WARNING'}, f"Failed to import FBX: {e}")

        # Download BVH if requested and available
        if props.export_bvh:
            bvh_asset = assets.get('bvh')
            if bvh_asset and bvh_asset.get('url'):
                bvh_url = bvh_asset['url']
                bvh_filename = f"mocai_{video_id}.bvh"
                bvh_path = os.path.join(temp_dir, bvh_filename)

                props.status_message = "Downloading BVH file..."
                self.report({'INFO'}, "Downloading BVH file...")

                if client.download_asset(bvh_url, bvh_path):
                    downloaded_files.append(('bvh', bvh_path))

                    # Import BVH into Blender
                    try:
                        bpy.ops.import_anim.bvh(filepath=bvh_path)
                        self.report({'INFO'}, f"BVH imported successfully!")
                    except Exception as e:
                        self.report({'WARNING'}, f"Failed to import BVH: {e}")

        # Download USDZ if requested and available
        if props.export_usd:
            usd_asset = assets.get('usd')
            if usd_asset and usd_asset.get('url'):
                usd_url = usd_asset['url']
                usd_filename = f"mocai_{video_id}.usdz"
                usd_path = os.path.join(temp_dir, usd_filename)

                props.status_message = "Downloading USDZ file..."
                self.report({'INFO'}, "Downloading USDZ file...")

                if client.download_asset(usd_url, usd_path):
                    downloaded_files.append(('usd', usd_path))

                    # Import USDZ into Blender
                    try:
                        bpy.ops.wm.usd_import(filepath=usd_path)
                        self.report({'INFO'}, f"USDZ imported successfully!")
                    except Exception as e:
                        self.report({'WARNING'}, f"Failed to import USDZ: {e}")

        if downloaded_files:
            files_str = ', '.join([f"{fmt.upper()}: {os.path.basename(path)}" for fmt, path in downloaded_files])
            props.status_message = f"Complete! Imported: {files_str}"
            self.report({'INFO'}, f"Successfully downloaded and imported {len(downloaded_files)} file(s)")
            return {'FINISHED'}
        else:
            self.report({'WARNING'}, "No files were downloaded")
            props.status_message = "No files available for download"
            return {'CANCELLED'}


class MOCAI_OT_cancel_processing(Operator):
    """Cancel current processing"""
    bl_idname = "mocai.cancel_processing"
    bl_label = "Cancel"
    bl_description = "Cancel current video processing"

    def execute(self, context):
        props = context.scene.mocai

        props.is_processing = False
        props.status_message = "Processing cancelled"

        self.report({'INFO'}, "Processing cancelled")

        return {'FINISHED'}


# Registration
classes = (
    MOCAI_OT_authenticate,
    MOCAI_OT_upload_video,
    MOCAI_OT_poll_status,
    MOCAI_OT_download_assets,
    MOCAI_OT_cancel_processing,
)


def register():
    """Register operators"""
    for cls in classes:
        bpy.utils.register_class(cls)


def unregister():
    """Unregister operators"""
    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)
